# -*- coding: utf-8 -*-
"""
Created on Wed Apr 21 08:38:11 2021

@author: Dyrvik
"""
import numpy as np
import matplotlib.pyplot as plt
import matplotlib.cm as cm
import matplotlib as mpl
import seaborn as sns
import pandas as pd
import os
from mpl_toolkits.axes_grid1 import Divider, Size


def adjust_box_widths(g, fac):
    """
    Adjust the widths of a seaborn-generated boxplot.
    """

    # iterating through Axes instances
    for ax in g.axes:

        # iterating through axes artists:
        for c in ax.get_children():

            # searching for mpl.patches.PathPatches
            if isinstance(c, mpl.patches.PathPatch):
                # getting current width of box:
                p = c.get_path()
                verts = p.vertices
                verts_sub = verts[:-1]
                xmin = np.min(verts_sub[:, 0])
                xmax = np.max(verts_sub[:, 0])
                xmid = 0.5*(xmin+xmax)
                xhalf = 0.5*(xmax - xmin)

                # setting new width of box
                xmin_new = xmid-fac*xhalf
                xmax_new = xmid+fac*xhalf
                verts_sub[verts_sub[:, 0] == xmin, 0] = xmin_new
                verts_sub[verts_sub[:, 0] == xmax, 0] = xmax_new

                # setting new width of median line
                for l in ax.lines:
                    if np.all(l.get_xdata() == [xmin, xmax]):
                        l.set_xdata([xmin_new, xmax_new])

"""Colourmaps"""
CM_SUBSECTION_GREENS = np.linspace(0.4, 1.0, 1000) 
COLOURS_GREENS = [ cm.Greens(x) for x in CM_SUBSECTION_GREENS ]

CM_SUBSECTION_ORANGES = np.linspace(0.3, 1.0, 1000) 
COLOURS_ORANGES = [ cm.Oranges(x) for x in CM_SUBSECTION_ORANGES ]

"""File handling"""
main_folder_path = os.path.dirname(os.path.abspath(__file__))
main_folder_path = '\\\\?\\' + main_folder_path
data_folder_path = main_folder_path
export_folder_path = main_folder_path
# =============================================================================
# export_folder_name = 'Plots'
# export_folder_path = os.path.join(main_folder_path,export_folder_name)
# if not os.path.exists(export_folder_path):
#     os.mkdir(export_folder_path) # create folder if non-existent
# =============================================================================

file_name = 'first measurements only.csv'
file_path = os.path.join(data_folder_path,file_name)
df = pd.read_csv(file_path) 
df_nonNeg = df.copy()
df_nonNeg['Current Density at 3.0 V [mA/cm2]'][df['Current Density at 3.0 V [mA/cm2]'] < 0] = np.nan

"""Some constants for graphical settings"""
lw = 3
lw_box = lw*1.5
lw_markers = lw
ticksize_major = 8
ticksize_minor = 8


"""Box swarm plot general graphic settings"""
palette = [COLOURS_GREENS[650], COLOURS_ORANGES[400]]
palette_w = ['w', 'w'] # for white faces on the stripplot
marker_size = 8
boxprops = {'linewidth': lw_box}
lineprops = {'linewidth': lw_box}
boxplot_kwargs = dict({'boxprops': boxprops, 'medianprops': lineprops,
                       'whiskerprops': lineprops, 'capprops': lineprops,
                       'width': 0.75})


"""Creating dummy dataframes for stripplot"""
# =============================================================================
# Necessary to obtain filled circles with same facecolor but different edgecolors
# =============================================================================
df_noALD = df.copy()
df_10ALD = df.copy()
df_noALD['Current Density at 3.0 V [mA/cm2]']= np.where( df['ALD']=='No ALD' ,  df_noALD['Current Density at 3.0 V [mA/cm2]'], np.NaN)
df_10ALD['Current Density at 3.0 V [mA/cm2]']= np.where( df['ALD']=='10 cycles' ,  df_10ALD['Current Density at 3.0 V [mA/cm2]'], np.NaN)

"""Plotting the box strip plot"""
# =============================================================================
# fig = plt.figure(0)
# fig.set_size_inches(16,8)
# fig.set_tight_layout(True)
# ax = plt.gca()
# =============================================================================
fig = plt.figure(figsize=(15, 7.5))
h = [Size.Fixed(1.0), Size.Scaled(1.), Size.Fixed(.2)]
v = [Size.Fixed(0.7), Size.Scaled(1.), Size.Fixed(.5)]
divider = Divider(fig, (0, 0, 1, 1), h, v, aspect=False)
# The width and height of the rectangle are ignored.
ax = plt.axes([0.15, 0.15, 0.85, 0.85])
order = ['No TFB', '0.5', '2', '5', '10']

"""Box plot"""
sns.set_palette(sns.color_palette(palette))
sns.boxplot(x='TFB concentration [mg/ml]', y='Current Density at 3.0 V [mA/cm2]', data=df_nonNeg, order=order, ax=ax, hue='ALD', saturation=1, fliersize=0, **boxplot_kwargs)

"""Strip plot"""
sns.set_palette(sns.color_palette(palette_w))
sns.stripplot(x='TFB concentration [mg/ml]', y='Current Density at 3.0 V [mA/cm2]', data=df_noALD, order=order, ax=ax, hue='ALD', size=marker_size, dodge=True, edgecolor=palette[0], linewidth=lw_markers)
sns.stripplot(x='TFB concentration [mg/ml]', y='Current Density at 3.0 V [mA/cm2]', data=df_10ALD, order=order, ax=ax, hue='ALD', size=marker_size, dodge=True, edgecolor=palette[1], linewidth=lw_markers)

"""Adjusting the boxes"""
adjust_box_widths(fig,0.9)

# Setting edgecolor with no facecolor
for i,artist in enumerate(ax.artists):
    # Set the linecolor on the artist to the facecolor, and set the facecolor to None
    col = artist.get_facecolor()
    artist.set_edgecolor(col)
    artist.set_facecolor('None')

    # Each box has 6 associated Line2D objects (to make the whiskers, fliers, etc.)
    # Loop over them here, and use the same colour as above
    for j in range(i*6,i*6+6):
        line = ax.lines[j]
        line.set_color(col)
        line.set_mfc(col)
        line.set_mec(col)

"""Legend"""
handles, labels = ax.get_legend_handles_labels()
ax.legend(handles[0:2], labels[0:2], frameon = False, title=False) # only plotting the boxplot legend
# Fixing face and edgecolor of legend
for legpatch in ax.get_legend().get_patches():
    col = legpatch.get_facecolor()
    legpatch.set_edgecolor('None')

"""Axis details"""
ax.set_yscale("log")
ax.yaxis.set_major_locator(mpl.ticker.LogLocator(base=100))
ax.yaxis.set_minor_locator(mpl.ticker.LogLocator(base=10))
ax.set_xlabel('TFB concentration [mg/ml]')
ax.set_ylabel('Current Density [$\mathregular{mA/cm^2}$]')

"""Annotation"""
ax.text(0.02, 0.02, 'Bias: 3 V',
        horizontalalignment='left',
        verticalalignment='bottom',
        transform=ax.transAxes)

"""Graphical details"""
plt.rcParams['font.size'] = 24
plt.rc('axes', labelsize=30)    # fontsize of the x and y labels
plt.rcParams['axes.linewidth'] = lw
plt.rcParams['lines.linewidth'] = lw

ax.xaxis.set_tick_params(which='major', size=ticksize_major, width=lw)
ax.xaxis.set_tick_params(which='minor', size=ticksize_minor, width=lw)
ax.yaxis.set_tick_params(which='major', size=ticksize_major, width=lw)
ax.yaxis.set_tick_params(which='minor', size=ticksize_minor, width=lw)

"""Exporting plots"""
plot_name = 'Plot Electron-only Box Strip J at 3V'
export_name_pdf = plot_name + '.pdf'
export_name_svg = plot_name + '.svg'
#export_name_eps = plot_name + '.eps'
pdf_path = os.path.join(export_folder_path,export_name_pdf)
svg_path = os.path.join(export_folder_path,export_name_svg)
#eps_path = os.path.join(export_folder_path,export_name_eps)
if len(pdf_path) > 255:
    warning_string = 'Warning: plot file path longer than 255 characters, downstream handling issues may occur. Path: ' + pdf_path
    print(warning_string)
fig.savefig(pdf_path, bbox_inches='tight')
fig.savefig(svg_path, bbox_inches='tight')
#fig.savefig(eps_path, bbox_inches='tight')
plt.close(fig)

